//
//  EquipmentPackages.m
//  EC145Gen
//
//  Created by Vadim on 4/22/14.
//
//

#import "EquipmentPackages.h"
#import "MedicalPackage.h"
#import "UtilityCategories.h"
#import "EquipmentFileUploader.h"

@implementation EquipmentPackages {
    NSString* serializeKey_;
    NSMutableArray* packages_;
    NSMutableArray* packageKeys_;
    int nextSavedPackageNumber_;
    NSString* nextPackageNumberKey_;
}

@synthesize packages=packages_;
@synthesize serializeKey=serializeKey_;

- (id)initWithSerializeKey:(NSString *)key {
    self = [super init];
    if (self) {
        [self resetForSerializeKey:key];
    }
    return self;
}

- (id)initWithPackages:(EquipmentPackages *)packages andSerializeKey:(NSString *)key {
    self = [super init];
    if (self) {
        serializeKey_ = [EquipmentPackages serializeKeyForKey:key];
        nextPackageNumberKey_ = [key stringByAppendingString:@"_LastSavedPackage"];
       [self copyFromPackage:packages];
    }
    return self;
}

- (void)resetForSerializeKey:(NSString *)key {
    serializeKey_ = [EquipmentPackages serializeKeyForKey:key];
    nextPackageNumberKey_ = [key stringByAppendingString:@"_LastSavedPackage"];
    nextSavedPackageNumber_ = [[NSUserDefaults standardUserDefaults] integerForKey:nextPackageNumberKey_];
    if (nextSavedPackageNumber_ == 0) nextSavedPackageNumber_ = 1;
    [self loadPackages];
}

- (void)copyFromPackage:(EquipmentPackages *)package {
    for (MedicalPackage* package in packages_) {
        [self removeObserverForPackage:package];
    }
    packages_ = [NSMutableArray array];
    NSDictionary* dict = [package serializeToDictionary];
    [self deserializeFromDictionary:dict];
    nextSavedPackageNumber_ = 1;
    for (MedicalPackage* package in packages_) {
        package.serializeKey = [self nextPackageKey];
        [package serialize];
    }
    [self savePackages];
}

+ (NSString*)serializeKeyForKey:(NSString*)key {
    return [NSString stringWithFormat:@"key__EquipmentPackages_%@", key];
}

+ (EquipmentPackages*)packageForKey:(NSString *)key {
    NSArray* array =  [[NSUserDefaults standardUserDefaults] arrayForKey:[self serializeKeyForKey:key]];
    if (array) {
        return [[EquipmentPackages alloc] initWithSerializeKey:key];
    }
    return nil;
}

- (void)reloadPackages {
    
     //NSLog(@"reloadPackages ");
    for (MedicalPackage* package in packages_) {
        [self removeObserverForPackage:package];
    }
    packages_ = [NSMutableArray array];
    [self loadPackagesFromResource];
    nextSavedPackageNumber_ = packages_.count+1;
    [[NSUserDefaults standardUserDefaults] setInteger:nextSavedPackageNumber_ forKey:nextPackageNumberKey_];
    [self savePackages];
}

- (void)loadPackages {
    
    //NSLog(@"loadPackages ");
    
    for (MedicalPackage* package in packages_) {
        [self removeObserverForPackage:package];
    }
    packages_ = [NSMutableArray array];
    NSDictionary* dict = [[NSUserDefaults standardUserDefaults] dictionaryForKey:serializeKey_];
    if (dict) {
        [self deserializeFromDictionary:dict];
    } else {
        [self loadPackagesFromResource];
        nextSavedPackageNumber_ = packages_.count+1;
        [[NSUserDefaults standardUserDefaults] setInteger:nextSavedPackageNumber_ forKey:nextPackageNumberKey_];
    }
}

- (void)deserializeFromDictionary:(NSDictionary*)dict {
    _configNumber = [[dict valueForKey:@"configNumber"] intValue];
    NSArray* packages = [dict valueForKey:@"packages"];
    for (NSString* packageKey in packages) {
        MedicalPackage* package = [[MedicalPackage alloc] initWithSerializeKey:packageKey];
        [packages_ addObject:package];
        [self addObserverForPackage:package];
    }
}

- (void)savePackages {
    [[NSUserDefaults standardUserDefaults] setValue:[self serializeToDictionary] forKey:serializeKey_];
}

- (NSDictionary*)serializeToDictionary {
    NSMutableDictionary* dict = [NSMutableDictionary new];
    [dict setValue:[NSNumber numberWithInt:_configNumber] forKey:@"configNumber"];
    NSMutableArray* array = [NSMutableArray array];
    for (MedicalPackage* package in packages_) {
        [package serialize];
        [array addObject:package.serializeKey];
    }
    [dict setValue:array forKey:@"packages"];
    return dict;
}

- (void)loadPackagesFromResource {
    
    
    //NSLog(@"start loadPackagesFromResource");
    
    
    NSString *fileName = [self packagesFileName];
    
    if(![[NSFileManager defaultManager] fileExistsAtPath:fileName])
    {
        
        //NSLog(@"1 achtung loadPackagesFromResource");
        
        
        
        // Copy file form the resource first
        NSString *path = [[NSBundle mainBundle] pathForResource:@"Packages" ofType:@"txt"];
        NSString* text = [NSString stringWithContentsOfFile:path encoding:NSUTF8StringEncoding error:nil];
        text  = [text stringByReplacingOccurrencesOfString:@"\t" withString:@" "];
        [text writeToFile:fileName atomically:YES encoding:NSUTF8StringEncoding error:nil];
    }
    // has worked
    NSString* text = [NSString stringWithContentsOfFile:fileName encoding:NSUTF8StringEncoding error:nil] ;
    


    
    
    NSCharacterSet* charSet = [NSCharacterSet characterSetWithCharactersInString:@"\n\r"];
    NSArray* components = [text componentsSeparatedByCharactersInSet:charSet];
    //NSLog(@"components: %@", components);
    
    MedicalPackage* newPackage;
    
    int blocker = 0;
    
    
    NSArray* nameComponents;
    
    NSString *downloadedEquipDataString = @"";
    
    
    NSString *downloadedRestrictionsString = @"";
    
  
    
    for (NSString* string in components) {
        
        if ([string isEmptyString]) continue;
        
        blocker = 0;
        

        
        //NSLog(@" string =  %@ ", string );
                
        
       if ([ string hasPrefix:@"cRestriction"]){
           
           blocker = 1; // do not write this line to equipment list
           
           
           downloadedRestrictionsString = [downloadedRestrictionsString stringByAppendingString:string];
           downloadedRestrictionsString = [downloadedRestrictionsString stringByAppendingString:@"|"];
           
       }
        
        
        
        
        if ([ string hasPrefix:@"cData"]){
            
            blocker = 1; // do not write this line to equipment list
            
            downloadedEquipDataString = [downloadedEquipDataString stringByAppendingString:string];
            downloadedEquipDataString = [downloadedEquipDataString stringByAppendingString:@"|"];
            
            
        }
        
        
        
        
        
        if (blocker == 0) {
            
            if ([[string uppercaseString] hasPrefix:@"PACKAGE"] || [[string uppercaseString] hasPrefix:@"-PACKAGE"] ) {
                
                
                
                NSArray* packageNameComponents = [string componentsSeparatedByString:@"="];
                NSString* packageKey = [self nextPackageKey];
                [[NSUserDefaults standardUserDefaults] removeObjectForKey:packageKey];
                newPackage = [[MedicalPackage alloc] initWithSerializeKey:packageKey];
                [newPackage deleteItem:[newPackage.items firstObject]]; // Delete empty item
                if (packageNameComponents.count > 1) {
                    newPackage.packageName = packageNameComponents[1];
                }
                newPackage.selected = ![string hasPrefix:@"-"];
                [self addPackage:newPackage];
            } else {
                
                
                
                
                
                NSArray* stationComponents = [string componentsSeparatedByString:@","];
                MedicalItem* item = [[MedicalItem alloc] init];
                item.selected = YES;
                float weight;
                for (NSInteger i = 0; i < stationComponents.count; i++) {
                    NSString* component=  stationComponents[i];
                    while ([component containsString:@"  "]) {
                        component = [stationComponents[i] stringByReplacingOccurrencesOfString:@"  " withString:@""];
                    }
                    if (i != 0 ) {
                        component = [component stringByReplacingOccurrencesOfString:@" " withString:@""];
                    }
                    switch (i) {
                        case 0:
                            if ([component hasPrefix:@"-"]) {
                                item.station = [component substringFromIndex:1];
                                item.selected = NO;
                            } else {
                                item.station = component;
                                item.selected = YES;
                            }
                            break;
                            
                        case 1:
                            weight = [component floatValue];
                            break;
                            
                        case 2:
                            item.weightInKg = [component isEqualToString:@"kg"];
                            
                            
                            
                            item.weight = weight; // Make sure weight is updated in right units
                            
                            
                            
                            break;
                            
                        case 3:
                            item.arm = [component floatValue];
                            break;
                            
                        case 4:
                            item.moment = [component floatValue];
                            break;
                            
                        case 5:
                            item.armLat = [component floatValue];
                            break;
                            
                        case 6:
                            item.momentLat = [component floatValue];
                            break;
                            
                        default:
                            break;
                    }
                }
                [newPackage addItem:item];              }
            
        }else{
            
            //do nothing
        }
    }// end loop
    
    
    NSMutableArray *configNamesArray = [[NSMutableArray alloc] init];
    
    
    for(int i=1; i<nameComponents.count; i++) {
        
        [configNamesArray addObject:nameComponents[i]];
    }
    
    
    //NSLog(@"saving configNamesArray  %@ ", configNamesArray );
    
    //NSLog(@"saving downloadedEquipDataString  =%@", downloadedEquipDataString );
    
    

    
    NSUserDefaults *defaults  = [NSUserDefaults standardUserDefaults];
    
    [defaults  setObject:configNamesArray      forKey:@"configNamesArray"];
    [defaults  setObject:downloadedEquipDataString      forKey:@"downloadedEquipDataString"];    
    [defaults  setObject:downloadedRestrictionsString      forKey:@"downloadedRestrictionsString"];
    
    
}

- (void)saveToFile {
    NSString* string = @"";
    for (MedicalPackage* package in packages_) {
        string = [string stringByAppendingString:[package convertToString]];
    }
    [string writeToFile:[self packagesFileName] atomically:YES encoding:NSUTF8StringEncoding error:nil];
    //NSLog(@"Written to file: %@", [self packagesFileName]);
}

- (NSString*)packagesFileName {
    NSArray *paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
    NSString *documentsDirectory = [paths objectAtIndex:0];
    
    //make a file name to write the data to using the documents directory:
    return [NSString stringWithFormat:@"%@/Packages_%@.txt", documentsDirectory, serializeKey_];
}


- (void)deletePackage:(MedicalPackage *)package {
    [self removeObserverForPackage:package];
    [packages_ removeObject:package];
    [self savePackages];
    self.totalWeight = self.totalWeight; // Trigger observers
}

- (MedicalPackage*)createNewPackage {
    NSString* packageName = [NSString stringWithFormat:@"Package %d", nextSavedPackageNumber_];
    MedicalPackage* newPackage = [[MedicalPackage alloc] initWithSerializeKey:[self nextPackageKey]];
    newPackage.packageName = packageName;
    newPackage.selected = YES;
    [self addPackage:newPackage];
    return newPackage;
}

- (void)addPackage:(MedicalPackage *)package {
    [packages_ addObject:package];
    [self addObserverForPackage:package];
    [self savePackages];
}

- (NSString*)nextPackageKey {
    NSString* key = [NSString stringWithFormat:@"%@_EquipmentPackage_%d", serializeKey_, nextSavedPackageNumber_];
    nextSavedPackageNumber_++;
    [[NSUserDefaults standardUserDefaults] setInteger:nextSavedPackageNumber_ forKey:nextPackageNumberKey_];
    return key;
}

- (void)setWeightUnits:(bool)isKg {
    for (MedicalPackage* package in packages_) {
        package.weightInKg = isKg;
    }
}

- (NSString*)totalWeightString {
    return [NSString stringFromFloat:self.totalWeight];
}

- (NSString*)totalArmString {
    return [NSString stringFromFloat:self.totalArm];
}

- (NSString*)totalArmLatString {
    return [NSString stringFromFloat:self.totalArmLat];
}

- (NSString*)totalMomentString {
    return [NSString stringFromFloat:self.totalMoment];
}

- (NSString*)totalMomentLatString {
    return [NSString stringFromFloat:self.totalMomentLat];
}

- (float)totalWeight {
    float total = 0;
    for (MedicalPackage* package in packages_) {
        if (package.selected)
            total += package.totalWeight;
    }
    return total;
}

- (float)totalMoment {
    float total = 0;
    for (MedicalPackage* package in packages_) {
        if (package.selected)
            total += package.totalMoment;
    }
    return total;
}

- (float)totalMomentLat {
    float total = 0;
    for (MedicalPackage* package in packages_) {
        if (package.selected)
            total += package.totalMomentLat;
    }
    return total;
}

- (float)totalArm {
    if (self.totalWeight) return self.totalMoment/self.totalWeight;
    return 0;
}

- (float)totalArmLat {
    if (self.totalWeight) return self.totalMomentLat/self.totalWeight;
    return 0;
}


- (void)addObserverForPackage:(MedicalPackage*)package {
    [package addObserver:self forKeyPath:@"totalWeight" options:NSKeyValueObservingOptionNew context:nil];
    [package addObserver:self forKeyPath:@"totalMoment" options:NSKeyValueObservingOptionNew context:nil];
    [package addObserver:self forKeyPath:@"totalMomentLat" options:NSKeyValueObservingOptionNew context:nil];
    [package addObserver:self forKeyPath:@"selected" options:NSKeyValueObservingOptionNew context:nil];
}

- (void)removeObserverForPackage:(MedicalPackage*)package {
    [package removeObserver:self forKeyPath:@"totalWeight"];
    [package removeObserver:self forKeyPath:@"totalMoment"];
    [package removeObserver:self forKeyPath:@"totalMomentLat"];
    [package removeObserver:self forKeyPath:@"selected"];
}

- (void)dealloc {
    for (MedicalPackage* package in packages_) {
        [self removeObserverForPackage:package];
    }
}

- (void)observeValueForKeyPath:(NSString *)keyPath ofObject:(id)object change:(NSDictionary *)change context:(void *)context {
    self.totalWeight = self.totalWeight; // Trigger the observer
    [self savePackages];
}

- (void)wipeOut {
    for (MedicalPackage* package in packages_) {
        [package wipeOut];
    }
    [[NSFileManager defaultManager] removeItemAtPath:[self packagesFileName] error:nil];
    [[NSUserDefaults standardUserDefaults] removeObjectForKey:serializeKey_];
}


#pragma mark- Upload/Download methods

- (void)downloadPackagesFileWithCompletionBlock:(void (^)(bool, NSString *))block {
    EquipmentFileUploader* uploader = [[EquipmentFileUploader alloc] initWithUploadURL:_uploadURL andDownloadURL:_downloadURL];
    [uploader downloadToFile:[self packagesFileName] withCompletionBlock:^(bool status, NSString *error) {
        if (status) {
            [self reloadPackages];
        }
        //NSLog(@"url: %@ status: %d", _downloadURL, status);
        if (block) {
            block(status, error);
        }
    }];
}

- (void)uploadPackagesFileWithCompletionBlock:(void (^)(bool, NSString *))block {
    EquipmentFileUploader* uploader = [[EquipmentFileUploader alloc] initWithUploadURL:_uploadURL andDownloadURL:_downloadURL];
    [self saveToFile];
    [uploader uploadFile:[self packagesFileName] withCompletionBlock:block];
}

@end
